using System;
using System.Collections.Specialized;
using System.Globalization;

using Borland.Eco.ModelValidation;
using Borland.Eco.UmlRt;
using Borland.Eco.Interfaces;
using Borland.Eco.Support;

namespace Borland.Eco.Examples.OclScanner
{
	///<summary>
	///An extension to the model validation mechanism to scan for escape sequences in OCL strings.
	///</summary>
	public class OclScanner: IModelValidator
	{
		///<summary>
		///Register validation class into validation engine.
		///</summary>
		public static void IDERegister()
		{
			ValidatorPlugins.RegisterPlugin(new OclScanner());
		}
		private string Scope_DerivationOcl = "DerivationOCL";
		private string Scope_Constraint = "Constraint";
		private string Scope_DefaultStringRepresentation = "DefaultStringRepresentation";
		private string Scope_DerivationExpression = "DerivationExpression";
		
		private IEcoTypeSystem m_model;
		private ValidationReport m_Reports;

		
		private void ValidateExpression(string scopeName, IModelElement element, IModelElement context, string expression)
		{
			IClassifier validationContext;
			
			if (context is IClass)
				validationContext = context as IClassifier;
			else if (context is IAttribute)
				validationContext = m_model.GetClassifierByName((context as IAttribute).Type_.ObjectType.ToString());
			else
				return;
			
			// Do a simple check for escaped characters - extend check as needed.
			if (expression.IndexOf("\\") > 0)
			{
				string errorHeader =  string.Format("Possible OCL/regexp issue in {0}", scopeName);
				m_Reports.AddHint(InterfacesStringRes.sViolationMessage(errorHeader, string.Format("OCL Expression contains escape sequence ({0})", context.Name)), element);
			}
		}
		private void ValidateConstraints(IModelElement element, IModelElement context)
		{
			foreach (IConstraint constraint in element.Constraints)
			{
				string expression = constraint.Body.Body.Trim(new char[] {'"'});
				ValidateExpression(Scope_Constraint, element, context, expression);
			}
		}
		private IModelElement LocateModelElement(IClass class_, string featureName)
		{
			foreach (IFeature feature in class_.EcoClass.AllStructuralFeatures)
			{
				if (System.String.Compare(feature.Name, featureName, true, CultureInfo.InvariantCulture) == 0)
					return feature;
			}
			return null;
		}
		private void ValidateDerivationExpressions(IClass class_, string derivationExpressions)
		{
			if (derivationExpressions == null || derivationExpressions.Length == 0)
				return;
			StringCollection stringList = CommaTextSupport.CommaTextToStringCollection(derivationExpressions);
			
			foreach (string expression in stringList)
			{
				// Get the array of <name> and <value>
				string[] nameValue = expression.Split('=');
				// store <name>
				string derivationExpressionName = nameValue[0];
				// store <value>
				string derivationExpressionValue = nameValue[1];
				
				IModelElement modelElement = LocateModelElement(class_, derivationExpressionName);
				
				if ((modelElement == null) || (EcoTaggedValues.DerivationOCL.GetValue(modelElement.TaggedValues, "") == ""))
					return;
					
				IAttribute attribute = modelElement as IAttribute;
				if (attribute != null && attribute.EcoAttribute.IsDerived)		
					ValidateExpression(Scope_DerivationExpression, class_, class_, derivationExpressionValue);

				IAssociationEnd associationEnd = modelElement as IAssociationEnd;
				if (associationEnd != null && attribute.EcoAttribute.IsDerived)		
					ValidateExpression(Scope_DerivationExpression, class_, class_, derivationExpressionValue);
			}
		}

		public void ValidateModel(IEcoTypeSystem model, ValidationArgs validationArgs)
		{
			m_model = model;
			m_Reports = validationArgs.Report;
		}

		public void ValidateClass(IClass class_, ValidationArgs validationArgs)
		{
			if (class_ == null) throw new ArgumentNullException("class_");
			// DefaultStringRepresentation
			string stringRepresentation = EcoTaggedValues.DefaultStringRepresentation.GetValue(class_.TaggedValues);
			if (stringRepresentation != null && stringRepresentation.Length > 0)
				ValidateExpression(Scope_DefaultStringRepresentation, class_, class_, stringRepresentation);
			// Constraints
			ValidateConstraints(class_, class_);
			// Derivation expressions
			ValidateDerivationExpressions(class_, EcoTaggedValues.DerivationExpressions.GetValue(class_.TaggedValues, ""));
		}

		public void ValidateAssociation(IAssociation association, ValidationArgs validationArgs)
		{
			if (association == null) throw new ArgumentNullException("association");
			// Constraints
			IModelElement context = association.AssociationClass == null ? (IModelElement)association : (IModelElement)association.AssociationClass;
			ValidateConstraints(association, context);
		}

		public void ValidateAssociationEnd(IAssociationEnd associationEnd, ValidationArgs validationArgs)
		{
			if (associationEnd == null) throw new ArgumentNullException("associationEnd");
			// DerivationOcl
			if ((EcoTaggedValues.DerivationOCL.GetValue(associationEnd.TaggedValues, "") != "") && associationEnd.EcoAssociationEnd.IsDerived)
				ValidateExpression(Scope_DerivationOcl, associationEnd, associationEnd.EcoAssociationEnd.Class_, EcoTaggedValues.DerivationOCL.GetValue(associationEnd.TaggedValues, ""));
			// Constraints
			ValidateConstraints(associationEnd, associationEnd.EcoAssociationEnd.Class_);
		}
		
		public void ValidateAttribute(IAttribute attribute, ValidationArgs validationArgs)
		{
			if (attribute == null) throw new ArgumentNullException("attribute");
			// DerivationOcl
			if ((EcoTaggedValues.DerivationOCL.GetValue(attribute.TaggedValues, "") != "") && attribute.EcoAttribute.IsDerived)
				ValidateExpression(Scope_DerivationOcl, attribute, attribute.Owner, EcoTaggedValues.DerivationOCL.GetValue(attribute.TaggedValues, ""));
			// Constraints
			ValidateConstraints(attribute, attribute.Owner);			
		}

		public void PostValidation(IEcoTypeSystem model, ValidationArgs validationArgs)
		{
			// nothing
		}

		public void Reset()
		{
			m_model = null;
			m_Reports = null;
		}
	}	
}